import type { NextApiRequest, NextApiResponse } from "next";
import { getServerSession } from "next-auth";
import { authOptions } from "@/lib/authOptions";
import { prisma } from "@/lib/prisma";
import { z } from "zod";
import { sendTaskAssignmentEmail } from "@/lib/email";
import { getCurrentPeriodDate } from "@/lib/taskPeriods";

const CreateTaskSchema = z.object({
  title: z.string().min(3).max(200),
  description: z.string().min(3),
  assignedToId: z.preprocess(
    (val) => (val === "" ? null : val),
    z.string().cuid().nullable().optional()
  ),
  dueDate: z.string().datetime().optional().nullable(),
  startDate: z.string().datetime().optional().nullable(),
  isRecurring: z.boolean().optional().default(false),
  recurrencePattern: z
    .enum(["daily", "weekly", "monthly", "semestral"])
    .optional()
    .nullable(),
  // Scope fields
  scopeType: z
    .enum(["element", "location", "department"])
    .optional()
    .nullable(),
  selectedLocations: z.array(z.string().cuid()).optional().default([]),
  selectedElements: z.array(z.string().cuid()).optional().default([]),
  // Legacy fields (for backward compatibility)
  departmentId: z.string().cuid().optional().nullable(),
  locationId: z.string().cuid().optional().nullable(),
  elementId: z.string().cuid().optional().nullable(),
  attachments: z.array(z.string()).optional().default([]),
});

export default async function handler(
  req: NextApiRequest,
  res: NextApiResponse
) {
  const session = await getServerSession(req, res, authOptions);
  if (!session?.user?.id) {
    return res.status(401).json({ ok: false, error: "Unauthorized" });
  }

  const role = (session.user as any)?.role?.toUpperCase() || "";
  const isAdmin = role === "ADMIN";
  const isTechnician = role === "TECHNICIAN";
  const userId = (session.user as any).id;

  if (req.method === "GET") {
    try {
      const whereClause: any = {};

      // Technicians only see their assigned tasks
      if (isTechnician) {
        whereClause.assignedToId = userId;
      }
      // Admins see all tasks

      // Note: Status filtering is done in application layer after computing period completion
      // because recurring tasks have completedAt = null (only periods are completed)
      const status = req.query.status as string;

      // Filter by search (title or description)
      const search = req.query.search as string;
      if (search) {
        whereClause.AND = [
          {
            OR: [
              { title: { contains: search, mode: "insensitive" } },
              { description: { contains: search, mode: "insensitive" } },
            ],
          },
        ];
      }

      // Filter by assigned technician (admin only)
      if (isAdmin && req.query.assignedToId) {
        const assignedToIdParam = req.query.assignedToId as string;
        if (assignedToIdParam === "unassigned") {
          whereClause.assignedToId = null;
        } else {
          whereClause.assignedToId = assignedToIdParam;
        }
      }

      // Build filter conditions for department/location/element
      const filterConditions: any[] = [];

      // Filter by department - show tasks that contain items from this department
      if (req.query.departmentId) {
        const deptId = req.query.departmentId as string;
        filterConditions.push({
          OR: [
            // Direct department match
            { departmentId: deptId },
            // Element-level task where element belongs to this department
            {
              scopeType: "element",
              element: {
                location: {
                  departmentId: deptId,
                },
              },
            },
            // Location-level task where location belongs to this department
            {
              scopeType: "location",
              location: {
                departmentId: deptId,
              },
            },
            // Department-level task matching this department
            {
              scopeType: "department",
              departmentId: deptId,
            },
          ],
        });
      }

      // Filter by location - show tasks that contain items from this location
      if (req.query.locationId) {
        const locId = req.query.locationId as string;
        filterConditions.push({
          OR: [
            // Direct location match
            { locationId: locId },
            // Element-level task where element belongs to this location
            {
              scopeType: "element",
              element: {
                locationId: locId,
              },
            },
            // Location-level task matching this location
            {
              scopeType: "location",
              locationId: locId,
            },
          ],
        });
      }

      // Filter by element - show tasks that contain this element
      if (req.query.elementId) {
        const elId = req.query.elementId as string;
        filterConditions.push({
          OR: [
            // Direct element match
            { elementId: elId },
            // Location-level task where this element is in the location
            {
              scopeType: "location",
              location: {
                elements: {
                  some: {
                    id: elId,
                  },
                },
              },
            },
          ],
        });
      }

      // Combine filter conditions with existing whereClause
      if (filterConditions.length > 0) {
        if (!whereClause.AND) {
          whereClause.AND = [];
        }
        whereClause.AND.push(...filterConditions);
      }

      // Filter by date range
      if (req.query.dateFrom || req.query.dateTo) {
        whereClause.createdAt = {};
        if (req.query.dateFrom) {
          whereClause.createdAt.gte = new Date(req.query.dateFrom as string);
        }
        if (req.query.dateTo) {
          const endDate = new Date(req.query.dateTo as string);
          endDate.setHours(23, 59, 59, 999);
          whereClause.createdAt.lte = endDate;
        }
      }

      // Filter by hasNotOk - tasks that have at least one item marked as NOT_OK
      const hasNotOk = req.query.hasNotOk as string;
      if (hasNotOk === "true") {
        whereClause.itemCompletions = {
          some: {
            status: "NOT_OK",
          },
        };
      } else if (hasNotOk === "false") {
        // Tasks that don't have any NOT_OK items
        whereClause.itemCompletions = {
          none: {
            status: "NOT_OK",
          },
        };
      }

      let tasks = await prisma.task.findMany({
        where: whereClause,
        include: {
          createdBy: {
            select: { id: true, name: true, email: true },
          },
          assignedTo: {
            select: { id: true, name: true, email: true },
          },
          department: {
            select: { id: true, name: true },
          },
          location: {
            select: { id: true, name: true },
          },
          element: {
            include: {
              location: {
                select: { id: true, name: true, departmentId: true },
              },
            },
          },
          periodCompletions: {
            select: {
              id: true,
              periodDate: true,
              completedAt: true,
              submittedAt: true,
            },
          },
        },
        orderBy: [
          { createdAt: "desc" }, // Newest first
        ],
      });

      // Additional filtering for tasks with selectedLocations/selectedElements
      if (
        req.query.departmentId ||
        req.query.locationId ||
        req.query.elementId
      ) {
        const deptId = req.query.departmentId as string | undefined;
        const locId = req.query.locationId as string | undefined;
        const elId = req.query.elementId as string | undefined;

        const filteredTasks = [];
        for (const task of tasks) {
          let matches = true; // Task already matched in Prisma query, so it matches by default

          // Check department filter - if task has selectedLocations/selectedElements, verify they match
          if (deptId) {
            // Task already matched in Prisma if it has direct departmentId or relations
            // But we need to check selectedLocations/selectedElements
            if (task.selectedLocationsJson) {
              try {
                const selectedLocations = JSON.parse(
                  task.selectedLocationsJson
                );
                const locationsInDept = await prisma.location.findMany({
                  where: {
                    id: { in: selectedLocations },
                    departmentId: deptId,
                  },
                  select: { id: true },
                });
                if (locationsInDept.length > 0) {
                  matches = true;
                } else if (!task.departmentId || task.departmentId !== deptId) {
                  // If task doesn't have direct match and selected locations don't match, check element relations
                  if (
                    !task.element?.location?.departmentId ||
                    task.element.location.departmentId !== deptId
                  ) {
                    matches = false;
                  }
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }

            // Check if task has selected elements in locations of this department
            if (matches && task.selectedElementsJson) {
              try {
                const selectedElements = JSON.parse(task.selectedElementsJson);
                const elementsInDept = await prisma.element.findMany({
                  where: {
                    id: { in: selectedElements },
                    location: {
                      departmentId: deptId,
                    },
                  },
                  select: { id: true },
                });
                if (elementsInDept.length > 0) {
                  matches = true;
                } else if (!task.departmentId || task.departmentId !== deptId) {
                  if (
                    !task.element?.location?.departmentId ||
                    task.element.location.departmentId !== deptId
                  ) {
                    matches = false;
                  }
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }
          }

          // Check location filter
          if (locId && matches) {
            // Task already matched in Prisma if it has direct locationId or element relation
            // But we need to check selectedLocations/selectedElements
            if (task.selectedLocationsJson) {
              try {
                const selectedLocations = JSON.parse(
                  task.selectedLocationsJson
                );
                if (selectedLocations.includes(locId)) {
                  matches = true;
                } else if (!task.locationId || task.locationId !== locId) {
                  if (
                    !task.element?.locationId ||
                    task.element.locationId !== locId
                  ) {
                    matches = false;
                  }
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }

            // Check if task has selected elements in this location
            if (matches && task.selectedElementsJson) {
              try {
                const selectedElements = JSON.parse(task.selectedElementsJson);
                const elementsInLoc = await prisma.element.findMany({
                  where: {
                    id: { in: selectedElements },
                    locationId: locId,
                  },
                  select: { id: true },
                });
                if (elementsInLoc.length > 0) {
                  matches = true;
                } else if (!task.locationId || task.locationId !== locId) {
                  if (
                    !task.element?.locationId ||
                    task.element.locationId !== locId
                  ) {
                    matches = false;
                  }
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }
          }

          // Check element filter
          if (elId && matches) {
            // Task already matched in Prisma if it has direct elementId
            // But we need to check selectedElements
            if (task.selectedElementsJson) {
              try {
                const selectedElements = JSON.parse(task.selectedElementsJson);
                if (selectedElements.includes(elId)) {
                  matches = true;
                } else if (!task.elementId || task.elementId !== elId) {
                  matches = false;
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }

            // Check if task is department-level and element is in any selected location
            if (
              matches &&
              task.scopeType === "department" &&
              task.selectedLocationsJson
            ) {
              try {
                const selectedLocations = JSON.parse(
                  task.selectedLocationsJson
                );
                const elementInLoc = await prisma.element.findFirst({
                  where: {
                    id: elId,
                    locationId: { in: selectedLocations },
                  },
                  select: { id: true },
                });
                if (elementInLoc) {
                  matches = true;
                } else if (!task.elementId || task.elementId !== elId) {
                  matches = false;
                }
              } catch (e) {
                // Invalid JSON, skip
              }
            }
          }

          if (matches) {
            filteredTasks.push(task);
          }
        }

        tasks = filteredTasks;
      }

      // For recurring tasks, compute current period completion status
      const tasksWithPeriodStatus = tasks.map((task) => {
        let currentPeriodCompletedAt: Date | null = null;

        if (
          task.isRecurring &&
          task.recurrencePattern &&
          task.periodCompletions
        ) {
          try {
            // Validate recurrence pattern before calling getCurrentPeriodDate
            const validPatterns = ["daily", "weekly", "monthly", "semestral"];
            if (!validPatterns.includes(task.recurrencePattern)) {
              console.error(
                `Invalid recurrence pattern: ${task.recurrencePattern}`
              );
            } else {
              const currentPeriodDate = getCurrentPeriodDate(
                task.recurrencePattern
              );
              const normalizedCurrentPeriod = new Date(currentPeriodDate);
              normalizedCurrentPeriod.setUTCHours(0, 0, 0, 0);

              // Find completion for current period
              const currentPeriodCompletion = task.periodCompletions.find(
                (pc: any) => {
                  if (!pc || !pc.periodDate) return false;
                  try {
                    const normalizedPcDate = new Date(pc.periodDate);
                    normalizedPcDate.setUTCHours(0, 0, 0, 0);
                    return (
                      normalizedPcDate.getTime() ===
                      normalizedCurrentPeriod.getTime()
                    );
                  } catch (e) {
                    return false;
                  }
                }
              );

              if (currentPeriodCompletion?.completedAt) {
                currentPeriodCompletedAt = currentPeriodCompletion.completedAt;
              }
            }
          } catch (e) {
            console.error(
              "Error computing period completion:",
              e,
              "Task ID:",
              task.id
            );
          }
        }

        return {
          ...task,
          // For recurring tasks, use current period completion instead of task completion
          currentPeriodCompletedAt: currentPeriodCompletedAt
            ? currentPeriodCompletedAt.toISOString()
            : null,
          // Keep completedAt for backward compatibility, but for recurring tasks use current period completion
          completedAt: task.isRecurring
            ? currentPeriodCompletedAt
              ? currentPeriodCompletedAt.toISOString()
              : null
            : task.completedAt
            ? task.completedAt.toISOString()
            : null,
          // Ensure periodCompletions is always an array
          periodCompletions: task.periodCompletions || [],
        };
      });

      // Filter by status in application layer (after computing period completion for recurring tasks)
      let filteredTasks = tasksWithPeriodStatus;
      if (status === "completed") {
        filteredTasks = tasksWithPeriodStatus.filter(
          (task) => task.completedAt !== null
        );
      } else if (status === "pending") {
        filteredTasks = tasksWithPeriodStatus.filter(
          (task) => task.completedAt === null
        );
      }

      return res.status(200).json({
        ok: true,
        tasks: filteredTasks.map((task) => {
          let attachments = [];
          try {
            attachments = JSON.parse(task.attachmentsJson || "[]");
          } catch (e) {
            console.error("Error parsing attachments JSON:", e);
            attachments = [];
          }
          return {
            ...task,
            attachments,
          };
        }),
      });
    } catch (error: any) {
      console.error("Error fetching tasks:", error);
      console.error("Error stack:", error?.stack);
      return res.status(500).json({
        ok: false,
        error: "Failed to fetch tasks",
        details: error?.message || String(error),
        stack:
          process.env.NODE_ENV === "development" ? error?.stack : undefined,
      });
    }
  }

  if (req.method === "POST") {
    // Only admins can create tasks
    if (!isAdmin) {
      return res.status(403).json({ ok: false, error: "Forbidden" });
    }

    try {
      const parsed = CreateTaskSchema.safeParse(req.body);
      if (!parsed.success) {
        return res.status(400).json({
          ok: false,
          error: "Invalid input",
          details: parsed.error.flatten(),
        });
      }

      const {
        title,
        description,
        assignedToId,
        dueDate,
        startDate,
        isRecurring,
        recurrencePattern,
        scopeType,
        selectedLocations,
        selectedElements,
        departmentId,
        locationId,
        elementId,
        attachments,
      } = parsed.data;

      // Verify assigned user is a technician (if provided)
      if (assignedToId) {
        const assignedUser = await prisma.user.findUnique({
          where: { id: assignedToId },
        });
        if (!assignedUser) {
          return res
            .status(404)
            .json({ ok: false, error: "Assigned user not found" });
        }
        if (assignedUser.role.toUpperCase() !== "TECHNICIAN") {
          return res.status(400).json({
            ok: false,
            error: "Tasks can only be assigned to technicians",
          });
        }
      }

      // Verify optional relations if provided
      if (departmentId) {
        const dep = await prisma.department.findUnique({
          where: { id: departmentId },
        });
        if (!dep) {
          return res
            .status(400)
            .json({ ok: false, error: "Invalid departmentId" });
        }
      }

      if (locationId) {
        const loc = await prisma.location.findUnique({
          where: { id: locationId },
        });
        if (!loc) {
          return res
            .status(400)
            .json({ ok: false, error: "Invalid locationId" });
        }
        if (departmentId && loc.departmentId !== departmentId) {
          return res.status(400).json({
            ok: false,
            error: "locationId does not belong to the selected departmentId",
          });
        }
      }

      if (elementId) {
        const el = await prisma.element.findUnique({
          where: { id: elementId },
        });
        if (!el) {
          return res
            .status(400)
            .json({ ok: false, error: "Invalid elementId" });
        }
        if (locationId && el.locationId !== locationId) {
          return res.status(400).json({
            ok: false,
            error: "elementId does not belong to the selected locationId",
          });
        }
      }

      // Determine scope type if not explicitly provided
      let finalScopeType = scopeType;
      if (!finalScopeType) {
        if (elementId) finalScopeType = "element";
        else if (locationId) finalScopeType = "location";
        else if (departmentId) finalScopeType = "department";
        else finalScopeType = null; // General task
      }

      const task = await prisma.task.create({
        data: {
          title,
          description,
          assignedToId: assignedToId ? assignedToId : (null as any),
          createdById: userId,
          dueDate: dueDate ? new Date(dueDate) : null,
          startDate: startDate ? new Date(startDate) : null,
          isRecurring: isRecurring || false,
          recurrencePattern: recurrencePattern || null,
          scopeType: finalScopeType,
          selectedLocationsJson:
            selectedLocations && selectedLocations.length > 0
              ? JSON.stringify(selectedLocations)
              : null,
          selectedElementsJson:
            selectedElements && selectedElements.length > 0
              ? JSON.stringify(selectedElements)
              : null,
          departmentId: departmentId || null,
          locationId: locationId || null,
          elementId: elementId || null,
          attachmentsJson: JSON.stringify(attachments || []),
        } as any,
        include: {
          createdBy: {
            select: { id: true, name: true, email: true },
          },
          assignedTo: {
            select: { id: true, name: true, email: true },
          },
          department: {
            select: { id: true, name: true },
          },
          location: {
            select: { id: true, name: true },
          },
          element: {
            select: { id: true, name: true },
          },
        },
      });

      // Send email notification if task is assigned
      if (task.assignedTo && task.assignedTo.email) {
        try {
          await sendTaskAssignmentEmail(task.assignedTo.email, {
            id: task.id,
            title: task.title,
            description: task.description,
            dueDate: task.dueDate,
            assignedBy: task.createdBy
              ? {
                  name: task.createdBy.name || "",
                  email: task.createdBy.email || "",
                }
              : undefined,
          });
        } catch (error) {
          console.error("Failed to send task assignment email:", error);
          // Don't fail the request if email fails
        }
      }

      return res.status(201).json({
        ok: true,
        task: {
          ...task,
          attachments: JSON.parse(task.attachmentsJson || "[]"),
        },
      });
    } catch (error: any) {
      console.error("Error creating task:", error);
      return res.status(500).json({
        ok: false,
        error: "Failed to create task",
        details: error?.message,
      });
    }
  }

  return res.status(405).json({ ok: false, error: "Method not allowed" });
}
